// font.c

#include "main.h"
#include "font.h"
#include "gworld.h"

#define kWhitePixel 0
#define kGrayscaleTable 40
#define kNumFonts (picTinyFont-picFont+1)

SkittlesFontPtr font[kNumFonts] = {0};

static SkittlesFontPtr LoadFont( int pictID, unsigned char *letterMap )
{
	SkittlesFontPtr font;
	PicHandle fontPic;
	Byte *lastLine;
	Boolean success = false;
	int start, across, skip;

	font = (SkittlesFontPtr) NewPtrClear( sizeof( SkittlesFont ) );

	if( font )
	{
		fontPic = GetPicture( pictID );

		if( fontPic )
		{
			NewGWorld( &font->world, 8, &(**fontPic).picFrame, GetCTable(kGrayscaleTable), nil, 0 );
		
			if( font->world )
			{		
				font->pixMap = GetGWorldPixMap(font->world);
				LockPixels( font->pixMap );
				
				PrepareForGDrawing( font->world );
				DrawPicture( fontPic, &(**fontPic).picFrame ); 
				FinishGDrawing( font->world );
				
				font->rowBytes = GetPixRowBytes(font->pixMap);
				font->height = font->pixMap[0]->bounds.bottom;
				lastLine = (Byte *) GetPixBaseAddr(font->pixMap) + (font->rowBytes * (font->height - 1));
				across = 0;
				
				// Measure empty space between character breaks
				while( lastLine[across] == kWhitePixel ) across++;
				skip = across;
				
				
				success = true;

				// Measure character starts and widths
				while( *letterMap )
				{
					while( lastLine[across] != kWhitePixel ) across++;
					if( across > font->rowBytes ) 
					{
						success = false;
						break;
					}
					
					start = across;
					font->across[*letterMap] = across + (skip/2);

					while( lastLine[across] == kWhitePixel ) across++;		
					font->width [*letterMap] = across - start - skip;

					letterMap++;
				}
			}		
			
			ReleaseResource( (Handle)fontPic );
		}
	}
	
	if( success )
	{
		return font;
	}
	else
	{
		Error( errNoMemory, "\p(LoadFont)" );
		return nil;
	}
}

void InitFont( void ) 
{
	font[0] = LoadFont( picFont, (unsigned char*) "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!:,.()*?0123456789'|- " );
	font[1] = LoadFont( picHiScoreFont, (unsigned char*) "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*().,/-=_+<>?|'\":; " );
	font[2] = LoadFont( picContinueFont, (unsigned char*) "0123456789" );
	font[3] = LoadFont( picBalloonFont, (unsigned char*) "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*()-=_+;:,./<>? '\"" );
	font[4] = LoadFont( picZapFont, (unsigned char*) "0123456789*PS" );
	font[5] = LoadFont( picZapOutlineFont, (unsigned char*) "0123456789*" );
	font[6] = LoadFont( picVictoryFont, (unsigned char*) "AB" );
	font[7] = LoadFont( picBubbleFont, (unsigned char*) "*" );
	font[8] = LoadFont( picTinyFont, (unsigned char*) "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789,. " );
}

SkittlesFontPtr GetFont( int pictID )
{
	int fontID = pictID - picFont;
	
	if ((fontID < 0) || (fontID > kNumFonts)) Error( errUnknown, "\pGetFont" );
	if (font[fontID] == nil) Error( errUnknown, "\pfont == nil" );
	
	return font[fontID];
} 

void DisposeFont( SkittlesFontPtr font )
{
    font; 
    
/*	if( font )
	{
		if( font->world ) DisposeGWorld( font->world );
		
		DisposePtr( (Ptr) font );
	}*/
}

int GetTextWidth( SkittlesFontPtr font, const char *text )
{
	int width = 0;
	while( *text )
	{
		width += font->width[*text++];
	}
	
	return width;
}

